# Module: main
# Author: Roman V. M.
# Created on: 28.11.2014
# License: GPL v.3 https://www.gnu.org/copyleft/gpl.html
"""
Example video plugin that is compatible with Kodi 19.x "Matrix" and above
"""
import sys
from urllib.parse import urlencode, parse_qsl
import xbmc
import xbmcgui
import xbmcplugin
import requests
import time
import hashlib
import json
from urllib.parse import urlparse

_URL = sys.argv[0]
_HANDLE = int(sys.argv[1])


def get_proxy_url(type, url, headers={}):
    return 'http://{}:{}?'.format('localhost', 10789) + urlencode(
        {
            'type': type,
            'url': url,
            'headers': json.dumps(headers),
        })


base_params = {
    'pcode': '010110005',
    'version': '2.0.5',
    'devid': hashlib.md5(str(time.time()).encode()).hexdigest(),
    'sys': 'android',
    'sysver': 11,
    'brand': 'google',
    'model': 'Pixel_3_XL',
    'package': 'com.sevenVideo.app.android'
}

base_headers = {
    'User-Agent': 'okhttp/3.12.0',
}


def get_url(**kwargs):
    """
    Create a URL for calling the plugin recursively from the given set of keyword arguments.

    :param kwargs: "argument=value" pairs
    :return: plugin call URL
    :rtype: str
    """
    return '{}?{}'.format(_URL, urlencode(kwargs))


def list_categories():
    xbmcplugin.setPluginCategory(_HANDLE, '酷云77')
    xbmcplugin.setContent(_HANDLE, 'videos')

    list_item = xbmcgui.ListItem(label='搜索')
    list_item.setInfo('video', {
        'title': '搜索',
        'genre': '搜索',
        'mediatype': 'video'
    })
    url = get_url(action='search')
    is_folder = True
    xbmcplugin.addDirectoryItem(_HANDLE, url, list_item, is_folder)

    r = requests.get('http://api.kunyu77.com/api.php/provide/filter',
                     headers=base_headers)
    data = r.json()
    for category_id in data['data']:
        category_name = data['data'][category_id][0]['cat']
        list_item = xbmcgui.ListItem(label=category_name)
        list_item.setInfo('video', {
            'title': category_name,
            'genre': category_name,
            'mediatype': 'video'
        })
        url = get_url(action='listing',
                      category_id=category_id,
                      category_name=category_name,
                      page=1)
        is_folder = True
        xbmcplugin.addDirectoryItem(_HANDLE, url, list_item, is_folder)

    # xbmcplugin.addSortMethod(_HANDLE, xbmcplugin.SORT_METHOD_LABEL_IGNORE_THE)
    xbmcplugin.endOfDirectory(_HANDLE)


def list_videos(category_id, category_name, page):
    xbmcplugin.setPluginCategory(_HANDLE, category_name)
    xbmcplugin.setContent(_HANDLE, 'videos')

    r = requests.get('http://api.kunyu77.com/api.php/provide/searchFilter',
                     params={
                         'type_id': category_id,
                         'pagenum': page,
                         'pagesize': 24
                     },
                     headers=base_headers)
    data = r.json()
    for video in data['data']['result']:
        list_item = xbmcgui.ListItem(label=video['title'])
        list_item.setInfo('video', {
            'title': video['title'],
            'genre': category_name,
            'mediatype': 'video'
        })
        list_item.setArt({
            'thumb': video['videoCover'],
            'icon': video['videoCover'],
            'fanart': video['videoCover']
        })
        url = get_url(action='playlist',
                      video_id=video['id'],
                      category_name=category_name)
        is_folder = True
        xbmcplugin.addDirectoryItem(_HANDLE, url, list_item, is_folder)

    if page < data['data']['pagesize']:
        list_item = xbmcgui.ListItem(label='下一页')
        url = get_url(action='listing',
                      category_id=category_id,
                      category_name=category_name,
                      page=page + 1)
        is_folder = True
        xbmcplugin.addDirectoryItem(_HANDLE, url, list_item, is_folder)

    # xbmcplugin.addSortMethod(_HANDLE, xbmcplugin.SORT_METHOD_LABEL_IGNORE_THE)
    xbmcplugin.endOfDirectory(_HANDLE)


def get_tk(url, params, ts):
    keys = []
    for key in params:
        keys.append(key)
    keys.sort()

    src = urlparse(url).path
    for key in keys:
        src += str(params[key])
    src += str(ts)
    src += 'XSpeUFjJ'

    return hashlib.md5(src.encode()).hexdigest()


def playlist(video_id, category_name):
    xbmcplugin.setContent(_HANDLE, 'videos')

    ts = int(time.time())
    params = {
        'ids': video_id,
        'version': '2.0.4',
        'sj': ts,
    }
    headers = base_headers.copy()
    headers['t'] = str(ts)

    url = 'http://api.kunyu77.com/api.php/provide/videoDetail'
    headers['TK'] = get_tk(url, params, ts)
    r = requests.get(url, params=params, headers=headers)
    detail = r.json()['data']

    xbmcplugin.setPluginCategory(_HANDLE, detail['videoName'])

    url = 'http://api.kunyu77.com/api.php/provide/videoPlaylist'
    headers['TK'] = get_tk(url, params, ts)
    r = requests.get(url, params=params, headers=headers)
    episodes = r.json()['data']['episodes']

    for video in episodes:
        list_item = xbmcgui.ListItem(label=video['title'])
        list_item.setInfo(
            'video', {
                'title': video['title'],
                'country': detail['area'],
                'year': int(detail['year']),
                'episode': video['episode'],
                'cast': detail['actor'].split(' '),
                'director': detail['director'],
                'plot': detail['brief'],
                'genre': category_name,
                'mediatype': 'video'
            })
        list_item.setArt({
            'thumb': detail['videoCover'],
            'icon': detail['videoCover'],
            'fanart': detail['videoCover']
        })
        url = get_url(action='play', url=video['playurl'])
        is_folder = False
        xbmcplugin.addDirectoryItem(_HANDLE, url, list_item, is_folder)

    # xbmcplugin.addSortMethod(_HANDLE, xbmcplugin.SORT_METHOD_LABEL_IGNORE_THE)
    xbmcplugin.endOfDirectory(_HANDLE)


def play_video(url):
    url = get_proxy_url('main', url)
    play_item = xbmcgui.ListItem(path=url)
    xbmcplugin.setResolvedUrl(_HANDLE, True, listitem=play_item)


def search():
    kb = xbmc.Keyboard('', '请输入影片名称')
    kb.doModal()
    if not kb.isConfirmed():
        return
    keyword = kb.getText()

    xbmcplugin.setPluginCategory(_HANDLE, '搜索结果')
    xbmcplugin.setContent(_HANDLE, 'videos')

    url = 'http://api.kunyu77.com/api.php/provide/searchVideo'

    ts = int(time.time())
    params = base_params.copy()
    params['sj'] = ts
    params['searchName'] = keyword
    params['pg'] = 1

    headers = base_headers.copy()
    headers['t'] = str(ts)
    headers['TK'] = get_tk(url, params, ts)

    r = requests.get(url, params=params, headers=headers)
    data = r.json()
    for video in data['data']:
        list_item = xbmcgui.ListItem(label=video['videoName'])
        list_item.setInfo(
            'video', {
                'title': video['videoName'],
                'year': int(video['year']),
                'cast': video['starName'].split(','),
                'plot': video['brief'],
                'genre': video['tags'],
                'mediatype': 'video'
            })
        list_item.setArt({
            'thumb': video['videoCover'],
            'icon': video['videoCover'],
            'fanart': video['videoCover']
        })
        url = get_url(action='playlist',
                      video_id=video['id'],
                      category_name=video['tags'])
        is_folder = True
        xbmcplugin.addDirectoryItem(_HANDLE, url, list_item, is_folder)

    #xbmcplugin.addSortMethod(_HANDLE, xbmcplugin.SORT_METHOD_LABEL_IGNORE_THE)
    xbmcplugin.endOfDirectory(_HANDLE)


def router(paramstring):
    params = dict(parse_qsl(paramstring))
    if params:
        if params['action'] == 'listing':
            list_videos(params['category_id'], params['category_name'],
                        int(params['page']))
        elif params['action'] == 'playlist':
            playlist(params['video_id'], params['category_name'])
        elif params['action'] == 'play':
            play_video(params['url'])
        elif params['action'] == 'search':
            search()
        else:
            raise ValueError('Invalid paramstring: {}!'.format(paramstring))
    else:
        list_categories()


if __name__ == '__main__':
    router(sys.argv[2][1:])
